/**
 * Effect Toggle System
 * Allows users to enable/disable individual effects without resetting values
 */

// Track which effects are enabled
const effectStates = {};

/**
 * Initialize effect toggle buttons
 */
function initEffectToggles() {
  // Add toggle buttons to all panel headers that don't have them yet
  const panelHeaders = document.querySelectorAll('.panel-header[data-panel]');
  
  panelHeaders.forEach(header => {
    const panelName = header.getAttribute('data-panel');
    
    // Skip if already has controls
    if (header.querySelector('.panel-header-controls')) {
      return;
    }
    
    // Create controls container
    const controls = document.createElement('div');
    controls.className = 'panel-header-controls';
    
    // Create toggle button
    const toggleBtn = document.createElement('button');
    toggleBtn.className = 'effect-toggle-btn active';
    toggleBtn.setAttribute('data-effect', panelName);
    toggleBtn.setAttribute('title', 'Toggle effect');
    toggleBtn.innerHTML = `
      <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" fill="none" viewBox="0 0 24 24" stroke="currentColor">
        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z" />
      </svg>
    `;
    
    // Move chevron to controls
    const chevron = header.querySelector('.chevron');
    controls.appendChild(toggleBtn);
    if (chevron) {
      controls.appendChild(chevron);
    }
    
    header.appendChild(controls);
    
    // Initialize state
    effectStates[panelName] = true;
  });
  
  // Setup event listeners
  setupToggleListeners();
}

/**
 * Setup event listeners for toggle buttons
 */
function setupToggleListeners() {
  const toggleButtons = document.querySelectorAll('.effect-toggle-btn');
  
  toggleButtons.forEach(btn => {
    btn.addEventListener('click', (e) => {
      e.stopPropagation(); // Prevent panel collapse
      
      const effectName = btn.getAttribute('data-effect');
      const isActive = btn.classList.contains('active');
      
      if (isActive) {
        // Disable effect
        btn.classList.remove('active');
        btn.innerHTML = `
          <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" fill="none" viewBox="0 0 24 24" stroke="currentColor">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13.875 18.825A10.05 10.05 0 0112 19c-4.478 0-8.268-2.943-9.543-7a9.97 9.97 0 011.563-3.029m5.858.908a3 3 0 114.243 4.243M9.878 9.878l4.242 4.242M9.88 9.88l-3.29-3.29m7.532 7.532l3.29 3.29M3 3l3.59 3.59m0 0A9.953 9.953 0 0112 5c4.478 0 8.268 2.943 9.543 7a10.025 10.025 0 01-4.132 5.411m0 0L21 21" />
          </svg>
        `;
        effectStates[effectName] = false;
      } else {
        // Enable effect
        btn.classList.add('active');
        btn.innerHTML = `
          <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" fill="none" viewBox="0 0 24 24" stroke="currentColor">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z" />
          </svg>
        `;
        effectStates[effectName] = true;
      }
      
      // Trigger filter update
      if (typeof applyFilters === 'function') {
        applyFilters();
      }
    });
  });
}

/**
 * Check if an effect is enabled
 */
function isEffectEnabled(effectName) {
  return effectStates[effectName] !== false;
}

/**
 * Get adjustment value with toggle check
 */
function getAdjustmentValue(effectName, value, defaultValue = 0) {
  if (!isEffectEnabled(effectName)) {
    return defaultValue;
  }
  return value;
}

/**
 * Check if effect should be applied
 */
function shouldApplyEffect(effectName) {
  return isEffectEnabled(effectName);
}

// Make functions globally available
window.isEffectEnabled = isEffectEnabled;
window.getAdjustmentValue = getAdjustmentValue;
window.shouldApplyEffect = shouldApplyEffect;

// Auto-initialize
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', initEffectToggles);
} else {
  initEffectToggles();
}
