// ============================================
// AI NATURAL LANGUAGE PROCESSING
// ============================================

class VoiceAINLP {
  constructor() {
    this.intents = this.initializeIntents();
    this.context = {
      lastAction: null,
      lastValue: null,
      imageLoaded: false
    };
    // Initialize color presets
    this.colorPresets = typeof VoiceColorPresets !== 'undefined' ? new VoiceColorPresets() : null;
    // Initialize Gemini command processor
    this.geminiProcessor = typeof GeminiCommandProcessor !== 'undefined' ? new GeminiCommandProcessor() : null;
    this.useGeminiAI = true; // Flag to enable/disable Gemini AI
  }

  // Initialize AI intents with flexible patterns
  initializeIntents() {
    return {
      // Brightness intents
      brightness: {
        keywords: ['sáng', 'tối', 'độ sáng', 'brightness', 'light', 'dark', 'ánh sáng', 'chiếu sáng'],
        increase: ['sáng', 'tăng', 'lên', 'hơn', 'nhiều', 'cao', 'bright', 'increase', 'more', 'up'],
        decrease: ['tối', 'giảm', 'xuống', 'ít', 'thấp', 'dark', 'decrease', 'less', 'down'],
        action: 'brightness'
      },

      // Contrast intents
      contrast: {
        keywords: ['tương phản', 'contrast', 'độ tương phản', 'rõ nét', 'sắc nét'],
        increase: ['tăng', 'cao', 'mạnh', 'rõ', 'nét', 'increase', 'more', 'high'],
        decrease: ['giảm', 'thấp', 'nhạt', 'mờ', 'decrease', 'less', 'low'],
        action: 'contrast'
      },

      // Saturation intents
      saturation: {
        keywords: ['màu sắc', 'bão hòa', 'saturation', 'màu', 'color', 'rực rỡ', 'sống động'],
        increase: ['rực rỡ', 'sống động', 'đậm', 'tăng', 'nhiều', 'vivid', 'vibrant', 'more'],
        decrease: ['nhạt', 'ít', 'giảm', 'pale', 'less', 'desaturate'],
        action: 'saturation'
      },

      // Black & White
      blackWhite: {
        keywords: ['đen trắng', 'black white', 'grayscale', 'xám', 'đen', 'trắng', 'bỏ màu', 'không màu'],
        action: 'blackWhite'
      },

      // Restore color
      restoreColor: {
        keywords: ['màu', 'color', 'khôi phục', 'trả lại', 'bật lại', 'restore', 'colorize'],
        context: ['đen trắng', 'grayscale'],
        action: 'restoreColor'
      },

      // Blur
      blur: {
        keywords: ['mờ', 'blur', 'làm mờ', 'nhòe', 'soft'],
        action: 'blur'
      },

      // Sharpen
      sharpen: {
        keywords: ['nét', 'sắc nét', 'sharpen', 'sharp', 'rõ', 'chi tiết', 'detail'],
        action: 'sharpen'
      },

      // Flip
      flip: {
        keywords: ['lật', 'flip', 'mirror', 'reverse', 'lật ảnh'],
        horizontal: ['ngang', 'horizontal', 'trái phải', 'left right'],
        vertical: ['dọc', 'vertical', 'trên dưới', 'up down'],
        action: 'flip'
      },

      // Rotate
      rotate: {
        keywords: ['xoay', 'quay', 'rotate', 'turn', 'nghiêng'],
        action: 'rotate'
      },

      // Temperature
      temperature: {
        keywords: ['nhiệt độ', 'temperature', 'ấm', 'lạnh', 'warm', 'cool', 'tone'],
        increase: ['ấm', 'warm', 'vàng', 'yellow', 'orange'],
        decrease: ['lạnh', 'cool', 'xanh', 'blue'],
        action: 'temperature'
      },

      // Sepia/Vintage
      sepia: {
        keywords: ['sepia', 'vintage', 'cổ điển', 'retro', 'cũ', 'xưa', 'old'],
        action: 'sepia'
      },

      // Invert
      invert: {
        keywords: ['đảo ngược', 'đảo màu', 'invert', 'negative', 'âm bản', 'ngược màu', 'đảo ngược màu'],
        action: 'invert'
      },

      // Reset
      reset: {
        keywords: ['reset', 'đặt lại', 'xóa', 'clear', 'khôi phục', 'ban đầu', 'gốc', 'original'],
        action: 'reset'
      },

      // Download
      download: {
        keywords: ['tải', 'download', 'lưu', 'save', 'xuất', 'export'],
        action: 'download'
      },

      // Canvas Controls - Zoom
      zoomIn: {
        keywords: ['phóng to', 'zoom in', 'to hơn', 'lớn hơn', 'phóng', 'zoom'],
        increase: ['phóng to', 'to', 'lớn', 'zoom in', 'in'],
        action: 'zoomIn'
      },

      zoomOut: {
        keywords: ['thu nhỏ', 'zoom out', 'nhỏ hơn', 'thu', 'out'],
        decrease: ['thu nhỏ', 'nhỏ', 'thu', 'zoom out', 'out'],
        action: 'zoomOut'
      },

      zoomReset: {
        keywords: ['zoom 100', 'zoom reset', 'zoom gốc', 'zoom ban đầu', 'fit screen', 'vừa màn hình'],
        action: 'zoomReset'
      },

      // Canvas Controls - Fit
      fitToScreen: {
        keywords: ['vừa màn hình', 'fit screen', 'fit to screen', 'full screen', 'toàn màn hình'],
        action: 'fitToScreen'
      },

      // Combo Commands - Smart Adjustments
      autoEnhance: {
        keywords: ['tự động', 'auto', 'enhance', 'cải thiện', 'tối ưu', 'làm đẹp'],
        action: 'autoEnhance'
      },

      brightenAndContrast: {
        keywords: ['sáng và tương phản', 'sáng tương phản', 'sáng rõ', 'sáng nét'],
        action: 'brightenAndContrast'
      },

      colorBalance: {
        keywords: ['cân bằng màu', 'balance color', 'điều chỉnh màu', 'sửa màu', 'màu chuẩn'],
        action: 'colorBalance'
      },

      clarifyDetails: {
        keywords: ['làm rõ chi tiết', 'rõ chi tiết', 'chi tiết', 'detail', 'clarity', 'sắc nét chi tiết'],
        action: 'clarifyDetails'
      },

      portraitEnhance: {
        keywords: ['chân dung đẹp', 'làm đẹp chân dung', 'portrait enhance', 'da đẹp'],
        action: 'portraitEnhance'
      },

      landscapeEnhance: {
        keywords: ['phong cảnh đẹp', 'làm đẹp phong cảnh', 'landscape enhance', 'cảnh đẹp'],
        action: 'landscapeEnhance'
      }
    };
  }

  // Main AI processing function
  async analyzeIntent(transcript) {
    const text = transcript.toLowerCase().trim();
    console.log('AI analyzing:', text);

    // Try Gemini AI first if enabled
    if (this.useGeminiAI && this.geminiProcessor && this.geminiProcessor.apiKey) {
      console.log('🤖 Using Gemini AI for command processing...');
      try {
        const result = await this.geminiProcessor.processCommand(text);
        
        if (result.success && result.command && result.command.type !== 'unknown') {
          console.log('✓ Gemini understood:', result.command.type);
          return {
            understood: true,
            intent: result.command.type,
            command: result.command,
            message: `✓ Hiểu rồi: ${result.command.description}`,
            confidence: 0.95,
            source: 'gemini'
          };
        } else {
          console.log('⚠️ Gemini did not understand, falling back to rule-based NLP');
        }
      } catch (error) {
        console.error('❌ Gemini error, falling back to rule-based NLP:', error);
      }
    }

    // Fallback to rule-based NLP
    console.log('📋 Using rule-based NLP...');

    // Check for color preset first (higher priority)
    if (this.colorPresets) {
      const presetKeywords = ['màu', 'color', 'preset', 'phong cách', 'style', 'look', 'tone', 'tông'];
      const hasPresetKeyword = presetKeywords.some(keyword => text.includes(keyword));
      
      if (hasPresetKeyword) {
        const preset = this.colorPresets.detectPreset(text);
        if (preset) {
          console.log('🎨 Color preset detected:', preset.name);
          return {
            understood: true,
            intent: 'colorPreset',
            command: {
              type: 'colorPreset',
              preset: preset,
              presetName: text,
              description: `Áp dụng preset "${preset.name}"`
            },
            message: `✓ Hiểu rồi: Áp dụng preset "${preset.name}"`,
            confidence: 0.95,
            source: 'rule-based'
          };
        }
      }
    }

    // Extract numbers from text
    const numbers = this.extractNumbers(text);
    const value = numbers.length > 0 ? numbers[0] : null;

    // Analyze intent
    const intent = this.detectIntent(text);
    
    if (!intent) {
      return {
        understood: false,
        message: `❓ Không hiểu: "${transcript}". Thử nói rõ hơn?`,
        suggestions: this.getSuggestions(text)
      };
    }

    // Build command
    const command = this.buildCommand(intent, value, text);
    
    if (command) {
      return {
        understood: true,
        intent: intent.action,
        command: command,
        message: `✓ Hiểu rồi: ${command.description}`,
        confidence: intent.confidence,
        source: 'rule-based'
      };
    }

    return {
      understood: false,
      message: `❓ Không chắc bạn muốn làm gì với "${transcript}"`,
      suggestions: this.getSuggestions(text)
    };
  }

  // Detect intent from text
  detectIntent(text) {
    let bestMatch = null;
    let highestScore = 0;

    for (const [intentName, intent] of Object.entries(this.intents)) {
      const score = this.calculateIntentScore(text, intent);
      
      if (score > highestScore) {
        highestScore = score;
        bestMatch = {
          name: intentName,
          action: intent.action,
          intent: intent,
          confidence: score
        };
      }
    }

    // Require minimum confidence
    if (highestScore < 0.3) {
      return null;
    }

    return bestMatch;
  }

  // Calculate intent matching score
  calculateIntentScore(text, intent) {
    let score = 0;
    const words = text.split(/\s+/);

    // Check main keywords
    for (const keyword of intent.keywords) {
      if (text.includes(keyword)) {
        score += 0.5;
      }
    }

    // Check increase/decrease keywords
    if (intent.increase) {
      for (const keyword of intent.increase) {
        if (text.includes(keyword)) {
          score += 0.3;
        }
      }
    }

    if (intent.decrease) {
      for (const keyword of intent.decrease) {
        if (text.includes(keyword)) {
          score += 0.3;
        }
      }
    }

    // Check context
    if (intent.context && this.context.lastAction) {
      for (const ctx of intent.context) {
        if (this.context.lastAction.includes(ctx)) {
          score += 0.2;
        }
      }
    }

    return Math.min(score, 1.0);
  }

  // Build executable command
  buildCommand(intent, value, text) {
    const action = intent.action;

    switch (action) {
      case 'brightness':
        return this.buildAdjustmentCommand('brightness', intent, value, text, 'độ sáng');

      case 'contrast':
        return this.buildAdjustmentCommand('contrast', intent, value, text, 'độ tương phản');

      case 'saturation':
        return this.buildAdjustmentCommand('saturation', intent, value, text, 'độ bão hòa');

      case 'blackWhite':
        return {
          type: 'toggle',
          target: 'blackWhite',
          value: true,
          description: 'Chuyển sang đen trắng'
        };

      case 'restoreColor':
        return {
          type: 'toggle',
          target: 'blackWhite',
          value: false,
          description: 'Khôi phục màu sắc'
        };

      case 'blur':
        return {
          type: 'adjustment',
          target: 'blur',
          value: value || 5,
          description: `Làm mờ ${value || 5}px`
        };

      case 'sharpen':
        return {
          type: 'adjustment',
          target: 'sharpen',
          value: value || 50,
          description: `Làm nét ${value || 50}%`
        };

      case 'flip':
        const isHorizontal = this.containsAny(text, intent.intent.horizontal);
        return {
          type: 'flip',
          target: isHorizontal ? 'horizontal' : 'vertical',
          description: `Lật ${isHorizontal ? 'ngang' : 'dọc'}`
        };

      case 'rotate':
        return {
          type: 'rotate',
          value: value || 90,
          description: `Xoay ${value || 90}°`
        };

      case 'temperature':
        return this.buildAdjustmentCommand('temperature', intent, value, text, 'nhiệt độ màu');

      case 'sepia':
        return {
          type: 'filter',
          target: 'sepia',
          description: 'Áp dụng hiệu ứng Sepia'
        };

      case 'invert':
        return {
          type: 'toggle',
          target: 'invert',
          description: 'Đảo ngược màu'
        };

      case 'reset':
        return {
          type: 'reset',
          description: 'Reset tất cả'
        };

      case 'download':
        return {
          type: 'download',
          description: 'Tải xuống ảnh'
        };

      // Combo Commands
      case 'autoEnhance':
        return {
          type: 'combo',
          target: 'autoEnhance',
          description: 'Tự động cải thiện ảnh',
          adjustments: [
            { target: 'brightness', value: 10 },
            { target: 'contrast', value: 15 },
            { target: 'saturation', value: 10 },
            { target: 'vibrance', value: 15 },
            { target: 'clarity', value: 10 }
          ]
        };

      case 'brightenAndContrast':
        return {
          type: 'combo',
          target: 'brightenAndContrast',
          description: 'Tăng sáng và tương phản hợp lý',
          adjustments: [
            { target: 'brightness', value: 15 },
            { target: 'contrast', value: 20 },
            { target: 'highlights', value: -5 },
            { target: 'shadows', value: 10 }
          ]
        };

      case 'colorBalance':
        return {
          type: 'combo',
          target: 'colorBalance',
          description: 'Cân bằng màu da',
          adjustments: [
            { target: 'temperature', value: 5 },
            { target: 'tint', value: 0 },
            { target: 'saturation', value: 5 },
            { target: 'vibrance', value: 10 }
          ]
        };

      case 'clarifyDetails':
        return {
          type: 'combo',
          target: 'clarifyDetails',
          description: 'Làm rõ chi tiết',
          adjustments: [
            { target: 'clarity', value: 25 },
            { target: 'sharpen', value: 30 },
            { target: 'contrast', value: 10 },
            { target: 'highlights', value: -5 }
          ]
        };

      case 'portraitEnhance':
        return {
          type: 'combo',
          target: 'portraitEnhance',
          description: 'Làm đẹp chân dung',
          adjustments: [
            { target: 'brightness', value: 10 },
            { target: 'contrast', value: 5 },
            { target: 'saturation', value: 5 },
            { target: 'temperature', value: 10 },
            { target: 'tint', value: 3 },
            { target: 'highlights', value: 5 },
            { target: 'shadows', value: -5 },
            { target: 'vibrance', value: 10 },
            { target: 'clarity', value: -10 }
          ]
        };

      case 'landscapeEnhance':
        return {
          type: 'combo',
          target: 'landscapeEnhance',
          description: 'Làm đẹp phong cảnh',
          adjustments: [
            { target: 'brightness', value: 5 },
            { target: 'contrast', value: 20 },
            { target: 'saturation', value: 25 },
            { target: 'vibrance', value: 30 },
            { target: 'clarity', value: 20 },
            { target: 'highlights', value: -5 },
            { target: 'shadows', value: 10 }
          ]
        };

      case 'zoomIn':
        return {
          type: 'zoomIn',
          description: 'Phóng to'
        };

      case 'zoomOut':
        return {
          type: 'zoomOut',
          description: 'Thu nhỏ'
        };

      case 'zoomReset':
        return {
          type: 'zoomReset',
          description: 'Reset zoom'
        };

      case 'fitToScreen':
        return {
          type: 'fitToScreen',
          description: 'Vừa màn hình'
        };

      default:
        return null;
    }
  }

  // Build adjustment command (brightness, contrast, etc.)
  buildAdjustmentCommand(target, intent, value, text, label) {
    const isIncrease = this.containsAny(text, intent.intent.increase || []);
    const isDecrease = this.containsAny(text, intent.intent.decrease || []);

    let finalValue = value || 20; // Default value
    
    if (isDecrease) {
      finalValue = -Math.abs(finalValue);
    }

    const action = isDecrease ? 'Giảm' : 'Tăng';
    
    return {
      type: 'adjustment',
      target: target,
      value: finalValue,
      relative: !value, // If no number specified, it's relative
      description: `${action} ${label} ${Math.abs(finalValue)}${value ? '' : ' (mặc định)'}`
    };
  }

  // Extract numbers from text
  extractNumbers(text) {
    // Vietnamese number words
    const numberWords = {
      'một': 1, 'hai': 2, 'ba': 3, 'bốn': 4, 'năm': 5,
      'sáu': 6, 'bảy': 7, 'tám': 8, 'chín': 9, 'mười': 10,
      'mươi': 10, 'trăm': 100, 'không': 0, 'linh': 0
    };

    const numbers = [];

    // Extract digit numbers
    const digitMatches = text.match(/\d+/g);
    if (digitMatches) {
      numbers.push(...digitMatches.map(n => parseInt(n)));
    }

    // Extract word numbers (simple cases)
    const words = text.split(/\s+/);
    for (let i = 0; i < words.length; i++) {
      if (numberWords[words[i]]) {
        let num = numberWords[words[i]];
        
        // Handle "ba mươi" = 30
        if (i + 1 < words.length && words[i + 1] === 'mươi') {
          num *= 10;
          i++;
        }
        
        numbers.push(num);
      }
    }

    return numbers;
  }

  // Check if text contains any of the keywords
  containsAny(text, keywords) {
    return keywords.some(keyword => text.includes(keyword));
  }

  // Get suggestions based on partial match
  getSuggestions(text) {
    const suggestions = [];

    if (text.includes('sáng') || text.includes('tối')) {
      suggestions.push('Thử: "Làm sáng hơn" hoặc "Tối xuống 20"');
    }
    if (text.includes('màu')) {
      suggestions.push('Thử: "Màu sắc rực rỡ hơn" hoặc "Chuyển đen trắng"');
    }
    if (text.includes('mờ') || text.includes('nét')) {
      suggestions.push('Thử: "Làm mờ" hoặc "Làm nét hơn"');
    }

    if (suggestions.length === 0) {
      suggestions.push('Thử nói: "Làm sáng", "Đen trắng", "Làm mờ", "Lật ngang"');
    }

    return suggestions;
  }

  // Update context after action
  updateContext(action, value) {
    this.context.lastAction = action;
    this.context.lastValue = value;
  }
}

// Export
if (typeof module !== 'undefined' && module.exports) {
  module.exports = VoiceAINLP;
}
