// ============================================
// GEMINI AI COLOR PRESET GENERATOR
// ============================================

class GeminiColorGenerator {
  constructor(apiKey) {
    this.apiKey = apiKey || this.loadApiKey();
    this.apiEndpoint = typeof CONFIG !== 'undefined' && CONFIG.GEMINI_API_ENDPOINT 
      ? CONFIG.GEMINI_API_ENDPOINT 
      : 'https://generativelanguage.googleapis.com/v1beta/models/gemini-pro:generateContent';
    this.cache = new Map(); // Cache để tránh gọi API nhiều lần cho cùng 1 request
  }

  loadApiKey() {
    // Try to load from CONFIG first (from config.js)
    if (typeof CONFIG !== 'undefined' && CONFIG.GEMINI_API_KEY) {
      console.log('✓ Loaded Gemini API key from config');
      return CONFIG.GEMINI_API_KEY;
    }
    
    // Try to load from localStorage
    const savedKey = localStorage.getItem('geminiApiKey');
    if (savedKey) {
      console.log('✓ Loaded Gemini API key from localStorage');
      return savedKey;
    }
    
    console.log('⚠️ No Gemini API key found');
    return null;
  }

  saveApiKey(apiKey) {
    localStorage.setItem('geminiApiKey', apiKey);
    this.apiKey = apiKey;
  }

  async promptForApiKey() {
    // Use SweetAlert if available
    if (typeof Swal !== 'undefined') {
      const { value: apiKey } = await Swal.fire({
        title: '🔑 Gemini API Key',
        html: `
          <p>Để sử dụng tính năng tạo màu tự động bằng AI, bạn cần nhập Gemini API Key.</p>
          <p><a href="https://makersuite.google.com/app/apikey" target="_blank">Lấy API Key miễn phí tại đây</a></p>
        `,
        input: 'text',
        inputPlaceholder: 'Nhập Gemini API Key...',
        showCancelButton: true,
        confirmButtonText: 'Lưu',
        cancelButtonText: 'Hủy',
        inputValidator: (value) => {
          if (!value) {
            return 'Vui lòng nhập API Key!';
          }
        }
      });

      if (apiKey) {
        this.saveApiKey(apiKey);
        return apiKey;
      }
    } else {
      // Fallback to prompt
      const apiKey = prompt('Nhập Gemini API Key (lấy tại https://makersuite.google.com/app/apikey):');
      if (apiKey) {
        this.saveApiKey(apiKey);
        return apiKey;
      }
    }
    
    return null;
  }

  async generateColorPreset(description) {
    console.log('🤖 Gemini: Generating color preset for:', description);

    // Check cache first
    const cacheKey = description.toLowerCase().trim();
    if (this.cache.has(cacheKey)) {
      console.log('✓ Using cached preset');
      return this.cache.get(cacheKey);
    }

    // Check if API key exists
    if (!this.apiKey) {
      console.log('⚠️ No API key, prompting user...');
      const key = await this.promptForApiKey();
      if (!key) {
        return {
          success: false,
          error: 'Không có API key'
        };
      }
    }

    try {
      const prompt = this.buildPrompt(description);
      const response = await this.callGeminiAPI(prompt);
      
      if (response.success) {
        // Cache the result
        this.cache.set(cacheKey, response);
        return response;
      } else {
        return response;
      }
    } catch (error) {
      console.error('❌ Gemini API error:', error);
      return {
        success: false,
        error: error.message
      };
    }
  }

  buildPrompt(description) {
    return `Bạn là chuyên gia color grading cho ảnh. Người dùng muốn chỉnh ảnh theo phong cách: "${description}"

Hãy tạo ra các thông số điều chỉnh phù hợp cho phong cách này. Trả về ĐÚNG format JSON sau (không thêm markdown, không thêm giải thích):

{
  "name": "Tên phong cách (tiếng Việt)",
  "description": "Mô tả ngắn gọn",
  "adjustments": {
    "brightness": 0,
    "contrast": 0,
    "saturation": 0,
    "temperature": 0,
    "tint": 0,
    "highlights": 0,
    "shadows": 0,
    "vibrance": 0,
    "clarity": 0
  }
}

QUY TẮC:
- Tất cả giá trị từ -100 đến 100
- brightness: Độ sáng (-100 = tối, +100 = sáng)
- contrast: Độ tương phản (-100 = mờ, +100 = rõ nét)
- saturation: Độ bão hòa màu (-100 = xám, +100 = rực rỡ)
- temperature: Nhiệt độ màu (-100 = lạnh/xanh, +100 = ấm/vàng)
- tint: Sắc màu (-100 = xanh lá, +100 = hồng)
- highlights: Vùng sáng (-100 = tối, +100 = sáng)
- shadows: Vùng tối (-100 = tối hơn, +100 = sáng hơn)
- vibrance: Độ sống động (-100 = nhạt, +100 = sống động)
- clarity: Độ rõ nét (-100 = mờ, +100 = sắc nét)

VÍ DỤ:
- "Cinematic": brightness: -5, contrast: 15, saturation: -10, temperature: 5, tint: -3, highlights: -10, shadows: 10, vibrance: 5, clarity: 10
- "Vintage": brightness: 5, contrast: -10, saturation: -20, temperature: 15, tint: 5, highlights: 10, shadows: -5, vibrance: -15, clarity: -5
- "Warm": brightness: 5, contrast: 5, saturation: 10, temperature: 25, tint: 5, highlights: 0, shadows: 5, vibrance: 15, clarity: 0

Chỉ trả về JSON, không thêm gì khác.`;
  }

  async callGeminiAPI(prompt) {
    if (!this.apiKey) {
      throw new Error('No API key');
    }

    const url = `${this.apiEndpoint}?key=${this.apiKey}`;
    
    const requestBody = {
      contents: [{
        parts: [{
          text: prompt
        }]
      }],
      generationConfig: {
        temperature: 0.7,
        topK: 40,
        topP: 0.95,
        maxOutputTokens: 1024,
      }
    };

    console.log('📡 Calling Gemini API...');
    
    const response = await fetch(url, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify(requestBody)
    });

    if (!response.ok) {
      const errorData = await response.json();
      console.error('❌ Gemini API error:', errorData);
      
      // Check if it's an API key error
      if (response.status === 400 || response.status === 403) {
        // Clear invalid API key
        localStorage.removeItem('geminiApiKey');
        this.apiKey = null;
        throw new Error('API key không hợp lệ. Vui lòng nhập lại.');
      }
      
      throw new Error(`API error: ${response.status}`);
    }

    const data = await response.json();
    console.log('✓ Gemini response received');

    // Extract text from response
    const text = data.candidates[0]?.content?.parts[0]?.text;
    if (!text) {
      throw new Error('No response from Gemini');
    }

    // Parse JSON from response
    const preset = this.parsePresetFromResponse(text);
    
    if (preset) {
      return {
        success: true,
        preset: preset
      };
    } else {
      throw new Error('Failed to parse preset from response');
    }
  }

  parsePresetFromResponse(text) {
    try {
      // Remove markdown code blocks if present
      let cleanText = text.trim();
      cleanText = cleanText.replace(/```json\n?/g, '');
      cleanText = cleanText.replace(/```\n?/g, '');
      cleanText = cleanText.trim();

      // Parse JSON
      const preset = JSON.parse(cleanText);

      // Validate preset structure
      if (!preset.name || !preset.adjustments) {
        console.error('Invalid preset structure:', preset);
        return null;
      }

      // Validate adjustments
      const requiredKeys = ['brightness', 'contrast', 'saturation', 'temperature', 'tint', 'highlights', 'shadows', 'vibrance', 'clarity'];
      for (const key of requiredKeys) {
        if (!(key in preset.adjustments)) {
          console.error(`Missing adjustment key: ${key}`);
          return null;
        }
        
        // Clamp values to -100 to 100
        preset.adjustments[key] = Math.max(-100, Math.min(100, preset.adjustments[key]));
      }

      console.log('✓ Parsed preset:', preset.name);
      return preset;
    } catch (error) {
      console.error('Failed to parse JSON:', error);
      console.log('Raw text:', text);
      return null;
    }
  }

  clearCache() {
    this.cache.clear();
    console.log('✓ Cache cleared');
  }

  clearApiKey() {
    localStorage.removeItem('geminiApiKey');
    this.apiKey = null;
    console.log('✓ API key cleared');
  }
}

// Export
if (typeof module !== 'undefined' && module.exports) {
  module.exports = GeminiColorGenerator;
}
