// ============================================
// GEMINI AI COMMAND PROCESSOR
// ============================================

class GeminiCommandProcessor {
  constructor(apiKey) {
    this.apiKey = apiKey || this.loadApiKey();
    this.apiEndpoint = typeof CONFIG !== 'undefined' && CONFIG.GEMINI_API_ENDPOINT 
      ? CONFIG.GEMINI_API_ENDPOINT 
      : 'https://generativelanguage.googleapis.com/v1beta/models/gemini-pro:generateContent';
    this.cache = new Map();
    this.conversationHistory = [];
  }

  loadApiKey() {
    if (typeof CONFIG !== 'undefined' && CONFIG.GEMINI_API_KEY) {
      return CONFIG.GEMINI_API_KEY;
    }
    const savedKey = localStorage.getItem('geminiApiKey');
    if (savedKey) {
      return savedKey;
    }
    return null;
  }

  async processCommand(userInput) {
    console.log('🤖 Gemini: Processing command:', userInput);

    // Check cache
    const cacheKey = userInput.toLowerCase().trim();
    if (this.cache.has(cacheKey)) {
      console.log('✓ Using cached command');
      return this.cache.get(cacheKey);
    }

    if (!this.apiKey) {
      return {
        success: false,
        error: 'No API key'
      };
    }

    try {
      const prompt = this.buildCommandPrompt(userInput);
      const response = await this.callGeminiAPI(prompt);
      
      if (response.success) {
        this.cache.set(cacheKey, response);
        return response;
      } else {
        return response;
      }
    } catch (error) {
      console.error('❌ Gemini error:', error);
      return {
        success: false,
        error: error.message
      };
    }
  }

  buildCommandPrompt(userInput) {
    return `Bạn là trợ lý AI cho ứng dụng chỉnh sửa ảnh. Người dùng nói: "${userInput}"

Hãy phân tích yêu cầu và trả về lệnh thực thi dưới dạng JSON. Chỉ trả về JSON, không thêm markdown hay giải thích.

CÁC LOẠI LỆNH:

1. ADJUSTMENT (Điều chỉnh đơn):
{
  "type": "adjustment",
  "target": "brightness|contrast|saturation|temperature|tint|highlights|shadows|vibrance|clarity|blur|sharpen",
  "value": số từ -100 đến 100,
  "relative": true/false,
  "description": "Mô tả ngắn"
}

2. COMBO (Nhiều điều chỉnh):
{
  "type": "combo",
  "target": "tên combo",
  "description": "Mô tả",
  "adjustments": [
    {"target": "brightness", "value": 10},
    {"target": "contrast", "value": 15}
  ]
}

3. COLOR PRESET:
{
  "type": "colorPreset",
  "presetName": "tên preset hoặc mô tả",
  "description": "Mô tả"
}

4. TOGGLE (Bật/tắt):
{
  "type": "toggle",
  "target": "blackWhite|invert",
  "value": true/false,
  "description": "Mô tả"
}

5. FLIP (Lật):
{
  "type": "flip",
  "target": "horizontal|vertical",
  "description": "Mô tả"
}

6. ROTATE (Xoay):
{
  "type": "rotate",
  "value": góc (0-360),
  "description": "Mô tả"
}

7. ZOOM:
{
  "type": "zoomIn|zoomOut|zoomReset",
  "description": "Mô tả"
}

8. RESET:
{
  "type": "reset",
  "description": "Reset tất cả"
}

9. DOWNLOAD:
{
  "type": "download",
  "description": "Tải xuống ảnh"
}

VÍ DỤ:

Input: "làm sáng 30"
Output: {"type":"adjustment","target":"brightness","value":30,"relative":true,"description":"Tăng độ sáng 30"}

Input: "chuyển đen trắng"
Output: {"type":"toggle","target":"blackWhite","value":true,"description":"Chuyển sang đen trắng"}

Input: "màu cinematic"
Output: {"type":"colorPreset","presetName":"cinematic","description":"Áp dụng preset Cinematic"}

Input: "làm ảnh đẹp hơn"
Output: {"type":"combo","target":"autoEnhance","description":"Tự động cải thiện","adjustments":[{"target":"brightness","value":10},{"target":"contrast","value":15},{"target":"saturation","value":10},{"target":"vibrance","value":15}]}

Input: "phóng to"
Output: {"type":"zoomIn","description":"Phóng to"}

Input: "lật ngang"
Output: {"type":"flip","target":"horizontal","description":"Lật ngang"}

Input: "xoay 90 độ"
Output: {"type":"rotate","value":90,"description":"Xoay 90°"}

QUY TẮC:
- Luôn trả về JSON hợp lệ
- Không thêm markdown code blocks
- Không thêm giải thích
- Nếu không hiểu, trả về: {"type":"unknown","description":"Không hiểu yêu cầu"}
- Với yêu cầu phức tạp, ưu tiên dùng combo
- Với yêu cầu về màu sắc/phong cách, dùng colorPreset

Chỉ trả về JSON:`;
  }

  async callGeminiAPI(prompt) {
    if (!this.apiKey) {
      throw new Error('No API key');
    }

    const url = `${this.apiEndpoint}?key=${this.apiKey}`;
    
    const requestBody = {
      contents: [{
        parts: [{
          text: prompt
        }]
      }],
      generationConfig: {
        temperature: 0.3, // Lower temperature for more consistent JSON
        topK: 40,
        topP: 0.95,
        maxOutputTokens: 512,
      }
    };

    console.log('📡 Calling Gemini API...');
    
    const response = await fetch(url, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify(requestBody)
    });

    if (!response.ok) {
      const errorData = await response.json();
      console.error('❌ Gemini API error:', errorData);
      
      if (response.status === 400 || response.status === 403) {
        localStorage.removeItem('geminiApiKey');
        this.apiKey = null;
        throw new Error('API key không hợp lệ');
      }
      
      throw new Error(`API error: ${response.status}`);
    }

    const data = await response.json();
    console.log('✓ Gemini response received');

    const text = data.candidates[0]?.content?.parts[0]?.text;
    if (!text) {
      throw new Error('No response from Gemini');
    }

    const command = this.parseCommandFromResponse(text);
    
    if (command) {
      return {
        success: true,
        command: command
      };
    } else {
      throw new Error('Failed to parse command');
    }
  }

  parseCommandFromResponse(text) {
    try {
      let cleanText = text.trim();
      cleanText = cleanText.replace(/```json\n?/g, '');
      cleanText = cleanText.replace(/```\n?/g, '');
      cleanText = cleanText.trim();

      const command = JSON.parse(cleanText);

      // Validate command structure
      if (!command.type) {
        console.error('Invalid command: missing type');
        return null;
      }

      console.log('✓ Parsed command:', command.type);
      return command;
    } catch (error) {
      console.error('Failed to parse JSON:', error);
      console.log('Raw text:', text);
      return null;
    }
  }

  clearCache() {
    this.cache.clear();
    console.log('✓ Cache cleared');
  }

  clearHistory() {
    this.conversationHistory = [];
    console.log('✓ History cleared');
  }
}

// Export
if (typeof module !== 'undefined' && module.exports) {
  module.exports = GeminiCommandProcessor;
}
